/*
 * Copyright (c) 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Rick Macklem at The University of Guelph.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)nfs_subs.c  8.8 (Berkeley) 5/22/95
 */

#include <sys/cdefs.h>
__FBSDID("$FreeBSD$");

/*
 * These functions support the macros and help fiddle mbuf chains for
 * the nfs op functions. They do things like create the rpc header and
 * copy data between mbuf chains and uio lists.
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/bio.h>
#include <sys/buf.h>
#include <sys/proc.h>
#include <sys/mount.h>
#include <sys/vnode.h>
#include <sys/namei.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/malloc.h>
#include <sys/sysent.h>
#include <sys/syscall.h>

#include <vm/vm.h>
#include <vm/vm_object.h>
#include <vm/vm_extern.h>

#include <nfs4/rpcv2.h>
#include <nfs4/nfsproto.h>
#include <nfsserver/nfs.h>
#include <nfs4/xdr_subs.h>
#include <nfs4/nfs_common.h>

#include <netinet/in.h>

#include <machine/stdarg.h>

enum vtype nv3tov_type[8]= {
	VNON, VREG, VDIR, VBLK, VCHR, VLNK, VSOCK, VFIFO
};
nfstype nfsv3_type[9] = {
	NFNON, NFREG, NFDIR, NFBLK, NFCHR, NFLNK, NFSOCK, NFFIFO, NFNON
};

u_quad_t
nfs_curusec(void) 
{
	struct timeval tv;

	getmicrotime(&tv);
	return ((u_quad_t)tv.tv_sec * 1000000 + (u_quad_t)tv.tv_usec);
}

/*
 * copies mbuf chain to the uio scatter/gather list
 */
int
nfsm_mbuftouio(struct mbuf **mrep, struct uio *uiop, int siz, caddr_t *dpos)
{
	char *mbufcp, *uiocp;
	int xfer, left, len;
	struct mbuf *mp;
	long uiosiz, rem;
	int error = 0;

	mp = *mrep;
	mbufcp = *dpos;
	len = mtod(mp, caddr_t)+mp->m_len-mbufcp;
	rem = nfsm_rndup(siz)-siz;
	while (siz > 0) {
		if (uiop->uio_iovcnt <= 0 || uiop->uio_iov == NULL)
			return (EFBIG);
		left = uiop->uio_iov->iov_len;
		uiocp = uiop->uio_iov->iov_base;
		if (left > siz)
			left = siz;
		uiosiz = left;
		while (left > 0) {
			while (len == 0) {
				mp = mp->m_next;
				if (mp == NULL)
					return (EBADRPC);
				mbufcp = mtod(mp, caddr_t);
				len = mp->m_len;
			}
			xfer = (left > len) ? len : left;
#ifdef notdef
			/* Not Yet.. */
			if (uiop->uio_iov->iov_op != NULL)
				(*(uiop->uio_iov->iov_op))
				(mbufcp, uiocp, xfer);
			else
#endif
			if (uiop->uio_segflg == UIO_SYSSPACE)
				bcopy(mbufcp, uiocp, xfer);
			else
				copyout(mbufcp, uiocp, xfer);
			left -= xfer;
			len -= xfer;
			mbufcp += xfer;
			uiocp += xfer;
			uiop->uio_offset += xfer;
			uiop->uio_resid -= xfer;
		}
		if (uiop->uio_iov->iov_len <= siz) {
			uiop->uio_iovcnt--;
			uiop->uio_iov++;
		} else {
			uiop->uio_iov->iov_base =
			    (char *)uiop->uio_iov->iov_base + uiosiz;
			uiop->uio_iov->iov_len -= uiosiz;
		}
		siz -= uiosiz;
	}
	*dpos = mbufcp;
	*mrep = mp;
	if (rem > 0) {
		if (len < rem)
			error = nfs_adv(mrep, dpos, rem, len);
		else
			*dpos += rem;
	}
	return (error);
}

/*
 * Help break down an mbuf chain by setting the first siz bytes contiguous
 * pointed to by returned val.
 * This is used by the macros nfsm_dissect for tough
 * cases. (The macros use the vars. dpos and dpos2)
 */
void *
nfsm_disct(struct mbuf **mdp, caddr_t *dposp, int siz, int left)
{
	struct mbuf *mp, *mp2;
	int siz2, xfer;
	caddr_t ptr;
	void *ret;

	mp = *mdp;
	while (left == 0) {
		*mdp = mp = mp->m_next;
		if (mp == NULL)
			return NULL;
		left = mp->m_len;
		*dposp = mtod(mp, caddr_t);
	}
	if (left >= siz) {
		ret = *dposp;
		*dposp += siz;
	} else if (mp->m_next == NULL) {
		return NULL;
	} else if (siz > MHLEN) {
		panic("nfs S too big");
	} else {
		MGET(mp2, M_TRYWAIT, MT_DATA);
		mp2->m_next = mp->m_next;
		mp->m_next = mp2;
		mp->m_len -= left;
		mp = mp2;
		ptr = mtod(mp, caddr_t);
		ret = ptr;
		bcopy(*dposp, ptr, left);		/* Copy what was left */
		siz2 = siz-left;
		ptr += left;
		mp2 = mp->m_next;
		/* Loop around copying up the siz2 bytes */
		while (siz2 > 0) {
			if (mp2 == NULL)
				return NULL;
			xfer = (siz2 > mp2->m_len) ? mp2->m_len : siz2;
			if (xfer > 0) {
				bcopy(mtod(mp2, caddr_t), ptr, xfer);
				mp2->m_data += xfer;
				mp2->m_len -= xfer;
				ptr += xfer;
				siz2 -= xfer;
			}
			if (siz2 > 0)
				mp2 = mp2->m_next;
		}
		mp->m_len = siz;
		*mdp = mp2;
		*dposp = mtod(mp2, caddr_t);
	}
	return ret;
}

/*
 * Advance the position in the mbuf chain.
 */
int
nfs_adv(struct mbuf **mdp, caddr_t *dposp, int offs, int left)
{
	struct mbuf *m;
	int s;

	m = *mdp;
	s = left;
	while (s < offs) {
		offs -= s;
		m = m->m_next;
		if (m == NULL)
			return (EBADRPC);
		s = m->m_len;
	}
	*mdp = m;
	*dposp = mtod(m, caddr_t)+offs;
	return (0);
}

void *
nfsm_build_xx(int s, struct mbuf **mb, caddr_t *bpos)
{
	struct mbuf *mb2;
	void *ret;

	if (s > M_TRAILINGSPACE(*mb)) {
		MGET(mb2, M_TRYWAIT, MT_DATA);
		if (s > MLEN)
			panic("build > MLEN");
		(*mb)->m_next = mb2;
		*mb = mb2;
		(*mb)->m_len = 0;
		*bpos = mtod(*mb, caddr_t);
	}
	ret = *bpos;
	(*mb)->m_len += s;
	*bpos += s;
	return ret;
}

void *
nfsm_dissect_xx(int s, struct mbuf **md, caddr_t *dpos)
{
	int t1;
	char *cp2;
	void *ret;

	t1 = mtod(*md, caddr_t) + (*md)->m_len - *dpos;
	if (t1 >= s) {
		ret = *dpos;
		*dpos += s;
		return ret;
	}
	cp2 = nfsm_disct(md, dpos, s, t1); 
	return cp2;
}

int
nfsm_strsiz_xx(int *s, int m, struct mbuf **mb, caddr_t *bpos)
{
	u_int32_t *tl;

	tl = nfsm_dissect_xx(NFSX_UNSIGNED, mb, bpos);
	if (tl == NULL)
		return EBADRPC;
	*s = fxdr_unsigned(int32_t, *tl);
	if (*s > m)
		return EBADRPC;
	return 0;
}

int
nfsm_adv_xx(int s, struct mbuf **md, caddr_t *dpos)
{
	int t1;

	t1 = mtod(*md, caddr_t) + (*md)->m_len - *dpos;
	if (t1 >= s) {
		*dpos += s;
		return 0;
	}
	t1 = nfs_adv(md, dpos, s, t1);
	if (t1)
		return t1;
	return 0;
}

/*
 * Copy a string into mbufs for the hard cases...
 */
int
nfsm_strtmbuf(struct mbuf **mb, char **bpos, const char *cp, long siz)
{
	struct mbuf *m1 = NULL, *m2;
	long left, xfer, len, tlen;
	u_int32_t *tl;
	int putsize;

	putsize = 1;
	m2 = *mb;
	left = M_TRAILINGSPACE(m2);
	if (left > 0) {
		tl = ((u_int32_t *)(*bpos));
		*tl++ = txdr_unsigned(siz);
		putsize = 0;
		left -= NFSX_UNSIGNED;
		m2->m_len += NFSX_UNSIGNED;
		if (left > 0) {
			bcopy(cp, (caddr_t) tl, left);
			siz -= left;
			cp += left;
			m2->m_len += left;
			left = 0;
		}
	}
	/* Loop around adding mbufs */
	while (siz > 0) {
		MGET(m1, M_TRYWAIT, MT_DATA);
		if (siz > MLEN)
			MCLGET(m1, M_TRYWAIT);
		m1->m_len = NFSMSIZ(m1);
		m2->m_next = m1;
		m2 = m1;
		tl = mtod(m1, u_int32_t *);
		tlen = 0;
		if (putsize) {
			*tl++ = txdr_unsigned(siz);
			m1->m_len -= NFSX_UNSIGNED;
			tlen = NFSX_UNSIGNED;
			putsize = 0;
		}
		if (siz < m1->m_len) {
			len = nfsm_rndup(siz);
			xfer = siz;
			if (xfer < len)
				*(tl+(xfer>>2)) = 0;
		} else {
			xfer = len = m1->m_len;
		}
		bcopy(cp, (caddr_t) tl, xfer);
		m1->m_len = len+tlen;
		siz -= xfer;
		cp += xfer;
	}
	*mb = m1;
	*bpos = mtod(m1, caddr_t)+m1->m_len;
	return (0);
}

int
nfsm_strtom_xx(const char *a, int s, int m, struct mbuf **mb, caddr_t *bpos)
{
        u_int32_t *tl;
        int t1;

        if (s > m)
                return ENAMETOOLONG;
        t1 = nfsm_rndup(s) + NFSX_UNSIGNED;
        if (t1 <= M_TRAILINGSPACE(*mb)) {
                tl = nfsm_build_xx(t1, mb, bpos);
                *tl++ = txdr_unsigned(s);
                *(tl + ((t1 >> 2) - 2)) = 0;
                bcopy(a, tl, s);
        } else {
                t1 = nfsm_strtmbuf(mb, bpos, a, s);
                if (t1 != 0)
                        return t1;
        }
        return 0;
}

/*
 * Build/dissect XDR buffer with a format string.
 *
 *    u - unsigned
 *    h - hyper
 *    s - stringlength, string
 *    k - skip length (bytes)
 *    a - arraylength, componentlenght, array
 *    o - opaque fix length
 *    O - opaque var length in bytes
 */

void
nfsm_buildf_xx(struct mbuf **mb, caddr_t *bpos, char *fmt, ...)
{
	uint32_t *tl, t1, len, uval;
	uint64_t hval;
	va_list args;
	char *p, *which;

	va_start(args, fmt);
	for (which = fmt; *which != '\0'; which++)
		switch (*which) {
		case 'u':	/* Unsigned */
			tl = nfsm_build_xx(NFSX_UNSIGNED, mb, bpos);
			uval = va_arg(args, uint32_t);
			*tl++ = txdr_unsigned(uval);
			break;
		case 'h':	/* Hyper */
			tl = nfsm_build_xx(2 * NFSX_UNSIGNED, mb, bpos);
			hval = va_arg(args, uint64_t);
			txdr_hyper(hval, tl);
			break;
		case 'o':	/* Fixed-length opaque */
			len = va_arg(args, uint32_t);
			p = va_arg(args, char *);
			tl = nfsm_build_xx(nfsm_rndup(len), mb, bpos);
			bcopy(p, tl, len);
			break;
		case 'O':	/* Variable-length opaque */
		case 's':	/* String */
			len = va_arg(args, uint32_t);
			p = va_arg(args, char *);
			t1 = nfsm_strtom_xx(p, len, len, mb, bpos);
			break;
		case 'k':	/* Skip */
			len = va_arg(args, uint32_t);
			nfsm_build_xx(nfsm_rndup(len), mb, bpos);
			break;
		default:
			panic("Invalid buildf string %s[%c]", fmt, *which);
			break;
		}
	va_end(args);
}

int
nfsm_dissectf_xx(struct mbuf **md, caddr_t *dpos, char *fmt, ...)
{
	uint32_t *tl, t1, len, *uval;
	uint64_t *hval;
	va_list args;
	char *p, *which;

	va_start(args, fmt);
	for (which = fmt; *which != '\0'; which++)
		switch (*which) {
		case 'u':	/* Unsigned */
			tl = nfsm_dissect_xx(NFSX_UNSIGNED, md, dpos);
			if (tl == NULL)
				return (EBADRPC);
			uval = va_arg(args, uint32_t *);
			*uval = fxdr_unsigned(uint32_t, *tl++);
			break;
		case 'h':	/* Hyper */
			tl = nfsm_dissect_xx(2 * NFSX_UNSIGNED, md, dpos);
			if (tl == NULL)
				return (EBADRPC);
			hval = va_arg(args, uint64_t *);
			*hval = fxdr_hyper(tl);
			break;
		case 'o':	/* Fixed-length opaque */
			len = va_arg(args, uint32_t);
			p = va_arg(args, void *);
			tl = nfsm_dissect_xx(nfsm_rndup(len), md, dpos);
			if (tl == NULL)
				return (EBADRPC);
			bcopy(tl, p, len);
			break;
		case 'O':	/* Variable-length opaque */
		case 's':	/* String */
			len = va_arg(args, uint32_t);
			p = va_arg(args, char *);
			tl = nfsm_dissect_xx(nfsm_rndup(len), md, dpos);
			if (tl == NULL)
				return (EBADRPC);
			bcopy(tl, p, len);
			break;
		case 'k':	/* Skip bytes */
			len = va_arg(args, uint32_t);
			t1 = nfsm_adv_xx(nfsm_rndup(len), md, dpos);
			break;
		default:
			panic("Invalid dissectf string %s[%c]", fmt, *which);
			break;
		}
	va_end(args);

	return (0);
}
